/**
* \file: cairo-drawing.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* Example for drawing graphics primitives with cairo.
* \component: graphics demos
*
* \author: Jens Georg <jgeorg@de.adit-jv.com>
*
* \copyright (c) 2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
***********************************************************************/

#include <cairo.h>
#include <stdlib.h>
#include <stdio.h>

#include "cairo-font-handling.h"

#include "cairo-drawing.h"

static int example_drawing_state = 0;

static void draw_progress_bar(cairo_t *cr, double x, double y,
                              double width, double height,
                              double progress, const char *text);

static void draw_interface(cairo_t *cr, double width, double height);

/**
 * @brief Function to draw on a buffer using cairo
 * @param buffer mapped DRM buffer data
 * @param width of buffer
 * @param height of buffer
 * @param stride of buffer
 *
 * The purpose of this function is to demonstrate how to draw on an arbitrary
 * buffer using cairo.
 *
 * In this case, we are directly wrapping the passed DRM dumb buffer in a cairo
 * surface to work with.
 *
 * The following code serves as an entry point for the sample ui that will paint
 * a simple interface as seen in the old example on top of a vendor logo.
 */
void modeset_draw_cairo(uint8_t *buffer,
                        uint32_t width, uint32_t height,
                        uint32_t stride)
{
    cairo_surface_t *surface;
    cairo_surface_t *background;
    cairo_surface_t *logo;
    cairo_t *cr;
    int logo_width;
    int logo_height;

    /* First, we create the cairo surface from the mapped DRM buffer */
    surface = cairo_image_surface_create_for_data(buffer,
                                                  CAIRO_FORMAT_ARGB32,
                                                  width,
                                                  height,
                                                  stride);
    if (surface == NULL)
    {
        fprintf(stderr, "Failed to create cairo surface.\n");
        return;
    }

    /* Secondly, create a cairo context for the surface */
    cr = cairo_create(surface);

    /* Now clear the buffer with black */
    cairo_set_source_rgb(cr, 0.0, 0.0, 0.0);
    cairo_paint(cr);

    /* Now we paint a background image with reduced opacity.
     *
     * Images can be loaded directly into cairo surfaces and then used to draw
     * them on the target.
     */
    background = cairo_image_surface_create_from_png(IMAGES_DIR JAC);
    cairo_set_source_surface(cr, background, 0.0, 0.0);
    cairo_paint_with_alpha(cr, 0.3);
    cairo_surface_destroy(background);

    /* Draw yellow rectangle on top. This is a simple example of drawing a
     * geometrical figure. The first, the paint color is set, then the rectangle
     * is prepared and its outline painted using cairo_stroke().
     *
     * For an example of a filled rectangle refer to draw_progress_bar().
     */
    cairo_set_source_rgb(cr, 1.0, 1.0, 0.0);
    cairo_set_line_width(cr, 4.0);
    cairo_rectangle(cr, 4.0, 4.0, width - 8.0, height - 8.0);
    cairo_stroke(cr);

    /* draw logo bottom right */
    logo = cairo_image_surface_create_from_png( IMAGES_DIR BOSCH2);
    logo_width = cairo_image_surface_get_width(logo);
    logo_height = cairo_image_surface_get_height(logo);
    cairo_set_source_surface(cr, logo,
                             width - logo_width - 10.0,
                             height - logo_height - 10.0);
    cairo_paint(cr);
    cairo_surface_destroy(logo);

    /* Paint the rest of the interface */
    draw_interface(cr, width, height);

    /* We are done with painting, destroy the cairo context and surface */
    cairo_destroy(cr);
    cairo_surface_destroy(surface);
}

/*****************************************************************************/
/* Helper functions                                                          */
/*****************************************************************************/

/**
 * @brief Draw progress bar.
 * @param cr cairo context
 * @param x left position of progress bar
 * @param y top position of progress bar
 * @param width of progress bar
 * @param height of progress bar
 * @param progress actual progress of the progress bar. Value is clamped to [0..1]
 * @param text to show inside the progress bar.
 *
 * Some example code to draw a simple progress bar with red outline and a
 * text centered within.
 */
static void draw_progress_bar(cairo_t *cr, double x, double y,
                              double width, double height,
                              double progress, const char *text)
{
    cairo_text_extents_t extents;
    double ypos;

    if (progress < 0.0)
    {
        progress = 0.0;
    }

    if (progress > 1.0)
    {
        progress = 1.0;
    }

    cairo_set_source_rgb(cr, 1.0, 1.0, 1.0);
    cairo_rectangle(cr, x, y, width * progress, height);
    cairo_fill(cr);

    cairo_rectangle(cr, x, y, width, height);
    cairo_set_line_width(cr, 4.0);
    cairo_set_source_rgb(cr, 1.0, 0.0, 0.0);

    cairo_stroke(cr);

    /*
     * Center text within rectangle. This is done by retrieving the
     * on-screen dimensions of the text with cairo_text_extents() and do
     * aligning according to them.
     */
    cairo_text_extents(cr, text, &extents);
    ypos = (y + height) - (height - extents.height) / 2.0;

    cairo_move_to(cr, x + (width - extents.width) / 2.0, ypos);
    cairo_show_text(cr, text);
}

static void draw_interface(cairo_t *cr, double width, double height)
{
    cairo_font_face_t *bold_font;
    cairo_font_face_t *font;
    cairo_text_extents_t extents;
    char *text = NULL;

    /* Common part */


    /* Create normal font */
    font = cairo_font_from_ttf_file("/usr/share/fonts/ttf/DroidSans.ttf");

    /* Create pseudo-bold font from the original font file. This can be used
     * if there is no bold variant of the font */
    bold_font = cairo_font_from_ttf_file("/usr/share/fonts/ttf/DroidSans.ttf");
    cairo_ft_font_face_set_synthesize(bold_font, CAIRO_FT_SYNTHESIZE_BOLD);

    /* Start rendering some bold text in size 22px on the buffer */
    cairo_set_font_face(cr, bold_font);
    cairo_set_font_size(cr, 22.0);

    /* Header text */
    text = "PLATFORMVERSION";
    cairo_text_extents (cr, text, &extents);

    cairo_set_source_rgb(cr, 0.0, 0.0, 0.0);
    /* clear background of text */
    cairo_rectangle(cr,
                    (width - extents.width) / 2.0 - 10.0, 0.0,
                    extents.width + 10.0, extents.height);
    cairo_fill(cr);

    cairo_move_to(cr, (width - extents.width) / 2.0, 20.0);
    cairo_set_source_rgb(cr, 1.0, 0.0, 0.0);
    cairo_show_text(cr, text);

    /* Some static labels */
    cairo_move_to(cr, 30.0, 40.0);
    cairo_show_text(cr, "Source:");

    cairo_move_to(cr, width - 280.0, 40.0);
    cairo_show_text(cr, "System:");

    cairo_move_to(cr, 30.0, 40.0 + extents.height * 2.0);
    cairo_show_text(cr, "STATUS:");

    /* Switch font to non-bold and use different color for the "content" */
    cairo_set_font_face(cr, font);
    cairo_set_font_size(cr, 22.0);
    cairo_set_source_rgb(cr, 1.0, 1.0, 1.0);

    cairo_move_to(cr, 160.0, 40.0);
    cairo_show_text(cr, "USB Host");

    cairo_move_to(cr, 160.0, 40.0 + extents.height * 2.0);
    cairo_show_text(cr, "Initialising...");

    /* Specific part */
    switch (example_drawing_state)
    {
    case 0: {
        /* state 0 simulates some in-progress download with some labels and a
         * progerss bar, using simple text.
         */
        double progress_bar_width;

        cairo_set_source_rgb(cr, 1.0, 0.0, 0.0);
        cairo_move_to(cr, 50.0, 40.0 + extents.height * 5.0);
        cairo_show_text(cr, "AI_PRJ_SUZUKI_LINUX_14.0F306 -> AI_PRJ_SUZUKI_LINUX_14.0F202 0120/");

        cairo_set_source_rgb(cr, 1.0, 1.0, 1.0);
        cairo_move_to(cr, width - 140.0, 40.0);
        cairo_show_text(cr, "V0120");

        cairo_move_to(cr, 40.0, 40.0 + extents.height * 7.0);
        cairo_show_text(cr, "USBDownload:");

        cairo_move_to(cr, 40.0, 40.0 + extents.height * 8.5);
        cairo_show_text(cr, "SWUPD of current target:2");

        /* Draw progress bar */
        progress_bar_width = width * 0.6;
        draw_progress_bar(cr, (width - progress_bar_width) / 2.0, height - 200.0,
                          progress_bar_width, 60.0,
                          0.53, "copying 53/100");

        cairo_set_font_face(cr, bold_font);
        cairo_move_to(cr, 40.0, height - 100.0);
        cairo_show_text(cr, "FILE:");

        cairo_move_to(cr, 40.0, height - 100.0 + extents.height * 1.5);
        cairo_show_text(cr, "COMMAND:");

        cairo_set_font_face(cr, font);
        cairo_set_source_rgb(cr, 1.0, 1.0, 1.0);
        cairo_move_to(cr, 230.0, height - 100.0);
        cairo_show_text(cr, "/tmp/iso");

        cairo_move_to(cr, 230.0, height - 100.0 + extents.height * 1.5);
        cairo_show_text(cr, "script.lua2");
        break;
    }
    case 1:
        /* State 1 draws a "Download finished" note in russian to show rendering
         * of non-ascii texts */
        cairo_font_draw_nonascii_centered(cr, width, height);
        break;
    default:
        break;
    }

    /* Cleanup */
    cairo_font_face_destroy(bold_font);
    cairo_font_face_destroy(font);
}

void next_state(void)
{
    example_drawing_state++;
    example_drawing_state %= 2;
}
